﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace uCalcDemoCS
{
    public partial class Form1 : Form
    {        
        public Graphics g;
        public Pen myPen = new Pen(Color.Black);
        static int AlreadyDisplayedOnce = 0;
        static String NumericFormat = "G";

        static void MyNumericFormat(int Expr)
        {
            String ResultAsString = uc.ucArgStr(Expr, 1);		

			if(ResultAsString=="Inf" || ResultAsString=="NaN") {
				uc.ucReturnStr(Expr, ResultAsString);
			}
			else {
				double Result = Convert.ToDouble(ResultAsString);
				String FormattedResult = Result.ToString(NumericFormat);
                uc.ucReturnStr(Expr, FormattedResult);         
			}                       
        }

        static int MyErrorHandler(int t)
		{
			if(AlreadyDisplayedOnce == uc.ucFalse)
				MessageBox.Show("Error Handler message: "+ uc.ucErrorMessage(0, t) + '\n'
				+ "Offending symbol: " + uc.ucErrorSymbol(t) + '\n'
				+ "Error Location: " + uc.ucErrorLocation(t).ToString() + '\n' + '\n'
				+ "This message box will not be displayed for the next error. " + '\n'
				+ "Remove 'AlreadyDisplayedOnce = 1;' in the demo source code to change this. ",
                                  "uCalc FMP error demo");

			// Remove the line below if you want the message box to be
			// displayed every time there's an error.
			AlreadyDisplayedOnce = 1;

			return uc.ucAbort;
		}

		static void MyArea(int Expr)
		{
			double MyLength, MyWidth;			

			MyLength = uc.ucArg(Expr, 1);
			MyWidth = uc.ucArg(Expr, 2);			
   
			if(MyLength < 0) uc.ucRaiseErrorMessage(Expr, "Length cannot be negative");
			if(MyWidth  < 0) uc.ucRaiseErrorMessage(Expr, "Width cannot be negative");

			uc.ucReturn(Expr, MyLength * MyWidth);
		}

		static void MyIIf_Numeric(int Expr)
		{
			double Condition;
			int TruePart, FalsePart;

			Condition = uc.ucArg(Expr, 1);
			TruePart = uc.ucArgHandle(Expr, 2);
			FalsePart = uc.ucArgHandle(Expr, 3);

			if(Condition != 0)
			uc.ucReturn(Expr, uc.ucEvaluate(TruePart));
			else uc.ucReturn(Expr, uc.ucEvaluate(FalsePart));
		}

		static void MyIIf_String(int Expr)
		{
			double Condition;
			int TruePart, FalsePart;

			Condition = uc.ucArg(Expr, 1);
			TruePart = uc.ucArgHandle(Expr, 2);
			FalsePart = uc.ucArgHandle(Expr, 3);

			if(Condition != 0)
			uc.ucReturnStr(Expr, uc.ucEvaluateStr(TruePart));
			else uc.ucReturnStr(Expr, uc.ucEvaluateStr(FalsePart));
		}

		static void MyMsgBox(int Expr)
		{
            // Second arg is ignored
            MessageBox.Show(uc.ucArgStr(Expr, 1), uc.ucArgStr(Expr, 3));
			uc.ucReturn(Expr, 1);
		}

		static void MyAverage(int Expr)
		{
			int x;
			double Total = 0;

			for(x=1; x <= uc.ucArgCount(Expr); x++)
				Total = Total + uc.ucArg(Expr, x);        

			uc.ucReturn(Expr, Total / uc.ucArgCount(Expr));
		}

		static void MyLeft(int Expr)
		{            
            uc.ucReturnStr(Expr, uc.ucArgStr(Expr, 1).Substring(0, (int)uc.ucArg(Expr, 2)));
		}

		static void StringRepeat(int Expr)
		{
			int x;
			string TotalString = "";

			for(x = 1; x <= uc.ucArg(Expr, 2); x++)
				TotalString = TotalString + uc.ucArgStr(Expr, 1);
        
			uc.ucReturnStr(Expr, TotalString);
		}

		static void ucSum(int Expr)
		{
			int Expression, VarHandle;
			double Start, Finish, sStep;
			double x, Total;

            Expression = uc.ucArgHandle(Expr, 1);
			Start = uc.ucArg(Expr, 2);
			Finish = uc.ucArg(Expr, 3);
			sStep = uc.ucArg(Expr, 4);
			VarHandle = uc.ucArgHandle(Expr, 5);

			x = Start;
			Total = 0;
			while(x <= Finish)
			{
				uc.ucSetVariableValue(VarHandle, x);
				Total = Total + uc.ucEvaluate(Expression);
				x = x + sStep;
			}

			uc.ucReturn(Expr, Total);
		}

		static void ucSolve(int Expr)
		{
			int Expression, Variable;
			int Iterations = 0;
			double a, b, fa, fb, Value, tmp;

			Expression = uc.ucArgHandle(Expr, 1);
			a = uc.ucArg(Expr, 2);
			b = uc.ucArg(Expr, 3);
			Variable = uc.ucArgHandle(Expr, 4);
			Value = 0;

			uc.ucSetVariableValue(Variable, a);
			fa = uc.ucEvaluate(Expression);

			uc.ucSetVariableValue(Variable, b); 
			fb = uc.ucEvaluate(Expression);
   
			if(fb < fa) {tmp = a; a = b; b = tmp;}  // swap a, b
    
			while(Math.Abs(b - a) > 0.000000000000001)
			{
				uc.ucSetVariableValue(Variable, (a + b) / 2);
      
				Value = uc.ucEvaluate(Expression);
      
				if(Value == 0) {a = (a + b) / 2; break;}

				if(Value < 0) a = (a + b) / 2; else b = (a + b) / 2;

				Iterations = Iterations + 1;
				if(Iterations == 100) break;
			}
   
			if(Math.Abs(Value) > 0.0000000001) uc.ucRaiseErrorMessage(Expr, "Solution not found");
			uc.ucReturn(Expr, a);
		}

		void Test(string Expr, string Answer)
		{
            if (Answer != uc.ucEvalStr(Expr)) MessageBox.Show(Expr + '\n' + uc.ucEvalStr(Expr) + '\n' + Answer, "", 0);
		}

		void TestExpand(string Expr, string Answer)
		{
            if (Answer != uc.ucExpand(Expr)) MessageBox.Show(Expr + '\n' + uc.ucExpand(Expr) + '\n' + Answer, "", 0);
		}

        public Form1()
        {
            InitializeComponent();
        }

        private void btnEval_Click(object sender, EventArgs e)
        {
            txtResult.Text = uc.ucEvalStr(cmbExpression.Text);
        }

        private void btnDefine_Click(object sender, EventArgs e)
        {
            uc.ucDefine(cmbDefine.Text);
        }

        uc.ucErrorHandlerDelegate d_MyErrorHandler = MyErrorHandler;
        uc.ucFunctionDelegate d_MyNumericFormat = MyNumericFormat;        
        uc.ucFunctionDelegate d_MyArea = MyArea;
        uc.ucFunctionDelegate d_MyLeft = MyLeft;
        uc.ucFunctionDelegate d_MyMsgBox = MyMsgBox;
        uc.ucFunctionDelegate d_MyAverage = MyAverage;
        uc.ucFunctionDelegate d_MyIIf_Numeric = MyIIf_Numeric;
        uc.ucFunctionDelegate d_MyIIf_String = MyIIf_String;
        uc.ucFunctionDelegate d_StringRepeat = StringRepeat;
        uc.ucFunctionDelegate d_ucSum = ucSum;
        uc.ucFunctionDelegate d_ucSolve = ucSolve;
        private void Form1_Load(object sender, EventArgs e)
        {
            g = PictureBox1.CreateGraphics();
            for (int x = 0; x <= 5; x++) { chkFPU.SetItemCheckState(x, CheckState.Checked); }

            // Note: In the current 64-bit implementation of uCalc, functions that involve a function address or delegate,
            //       such as the ones in this Form1_Load section, are ignored.

            // The definitions here are for "callback" routines.  Each definition ends with the
            // name of a function, which can be found elsewhere in this file.            
            uc.ucSetOutput(d_MyNumericFormat);

            // The MyErrorHandler() function will be called whenever and wherever an
            // error (such as "Syntax Error") is raised.	
            uc.ucAddErrorHandler(d_MyErrorHandler);

            // MyArea() returns the product of multiplying the two arguments.
            // MyLeft() returns the left-most characters of a string.  The second argument represents the number of characters.
            // MsgBox() displays a message box.  The 1st argument is required.  The others are optional.

            uc.ucDefineFunction("Native: MyArea(Length, Width)", d_MyArea);
            uc.ucDefineFunction("Native: MyLeft(Text As String, Count) As String", d_MyLeft);
            uc.ucDefineFunction("Native: MyMsgBox(Prompt As String, Buttons = 0, Title As String = 'uCalc')", d_MyMsgBox);
            uc.ucDefineSyntax("MsgBox ::= MyMsgBox");

            // The three consecutive dots "..." mean that MyAverage can take any number of arguments
            // (however, because "x" is specified, there must be at least one argument).
            uc.ucDefineFunction("Native: MyAverage(x ...)", d_MyAverage);

            // Two versions of MyIIF can co-exist peacefully with the same name because
            // they are defined with different argument types (numeric vs string).
            // By passing the last two arguments ByExpr, the callback can chose to evaluate only one of of them.  
            uc.ucDefineFunction("Native: MyIIf(cond, ByExpr TruePart, ByExpr FalsePart)", d_MyIIf_Numeric);
            uc.ucDefineFunction("Native: MyIIf(cond, ByExpr TruePart As String, ByExpr FalsePart As String) As String", d_MyIIf_String);

            // The following defines the "*" operator so that MyString * n returns MyString repeated n times.
            // For instance, "He " * 3, would return "He He He".  20 represents the precedence level.  It is
            // arbitrarily set at the same level as that of the numeric "*" multiplication operator.
            uc.ucDefineOperator("Native: Precedence('*') {MyString As String} * {Number} As String", d_StringRepeat);

            // This definition is for a summation.  See the callback ucSum() routine for the
            // actual code, which runs a loop that adds up the total for the expression in
            // the first argument a number of times based on the second and third arguments.
            // For instance Sum(g^2+1, 1, 5, 1, g) returns 60 and Sum(x^2, 1, 10) returns 385.
            // The last two arguments are optional, so they default to 1 and x if omitted.
            //
            // The first argument is passed "ByExpr".  So instead of being evaluated before
            // being passed to the callback the way an ordinary argument would be, a handle
            // for the expression is passed so that the callback can evaluate it (in this
            // case numerous times).
            //
            // The last argument is passed "ByHandle".  This causes the callback to receive
            // a handle for the variable being passed, in such a way that it can be linked to
            // the summation counter, and integrated into the expression in the first argument.
            //
            // The actual function being defined is Sum_().  Then a syntax construct named
            // Sum() is defined in such a way that the last argument gets defined as a local
            // variable.  So if you evaluate Sum(x^2, 5, 10, 1, x), the local "x" in this
            // expression will not interfere with a pre-existing variable named x.  Also you
            // do not need to declare a variable ahead of time to use it as a counter for Sum().
            uc.ucDefineFunction("Native: Sum_(ByExpr Expr, Start, Finish, Step, ByHandle Var)", d_ucSum);
            uc.ucDefineSyntax("Sum({Expr}, {Start}, {Finish} [, {Step=1} [, {Var=x}]])"
                    + "::= Local({Var}, Sum_({Expr}, {Start}, {Finish}, {Step}, {Var}))");

            // The following routine solves an equation.
            // The callback code is based on the Bisection Method algorithm.
            // The concept here is very similar to that of Sum().
            // Two syntax constructs are defined.  The second one rearranges the equation
            // if it includes an equal sign.
            // For instance Solve(x^2 = 9+x) becomes Solve(x^2 - (9+x))
            // Solve(x^2 + 1 = 26) returns 5.
            // Solve(x^2 + 1 = 26, -1000, 0) returns -5.
            uc.ucDefineFunction("Native: Solve_(ByExpr Expr, a, b, ByHandle Var)", d_ucSolve);
            uc.ucDefineSyntax("Solve({Expr} [, {a=-100000000} [, {b=100000000} [, {Var=x}]]]) "
                            + "::= Local({Var}, Solve_({Expr}, {a}, {b}, {Var}))");
            uc.ucDefineSyntax("Solve({Left} = {Right} [, {etc}]) ::= Solve({Left}-({Right}) {etc: , {etc}})");

        }

        private void chkFPU_SelectedIndexChanged(object sender, EventArgs e)
        {
            /*
            switch (chkFPU.SelectedIndex)
            {
                case 0: uc.ucFPU(uc.uc_ToggleFPU, uc.uc_FPU_Mask_InvalidOp); break;
                case 1: uc.ucFPU(uc.uc_ToggleFPU, uc.uc_FPU_Mask_DenormalOp); break;
                case 2: uc.ucFPU(uc.uc_ToggleFPU, uc.uc_FPU_Mask_ZeroDivide); break;
                case 3: uc.ucFPU(uc.uc_ToggleFPU, uc.uc_FPU_Mask_Overflow); break;
                case 4: uc.ucFPU(uc.uc_ToggleFPU, uc.uc_FPU_Mask_Underflow); break;
                case 5: uc.ucFPU(uc.uc_ToggleFPU, uc.uc_FPU_Mask_PrecisionLoss); break;
            }
             */
        }
         
        private void btnSummation_Click(object sender, EventArgs e)
        {           
            int ExprHandle, xHandle;
            double x = 1, SumTotal = 0, SumMax;
            System.Diagnostics.Stopwatch MyStopWatch = new System.Diagnostics.Stopwatch();

            xHandle = uc.ucDefineVariable("x");
            ExprHandle = uc.ucParse(txtSumExpression.Text);
            SumMax = System.Convert.ToDouble(txtSumMax.Text);

            MyStopWatch.Start();
            while (x <= SumMax)
            {                
                uc.ucSetVariableValue(xHandle, x);
                SumTotal = SumTotal + uc.ucEvaluate(ExprHandle);
                x++;
            }
            MyStopWatch.Stop();

            txtSumResult.Text = System.Convert.ToString(SumTotal);
            lblElapsed.Text = "Elapsed Time:  " + System.Convert.ToString((double)MyStopWatch.ElapsedMilliseconds / 1000);

            uc.ucReleaseItem(ExprHandle);
            uc.ucReleaseItem(xHandle);
        }

        public float OldX, OldY;
        private void CartesianLineTo(float x, float y)
        {
            float NewX, NewY;

            if (x == -10) { OldX = 0; OldY = 0; }

            NewX = PictureBox1.Size.Width * (x + 10) / 20;
            NewY = PictureBox1.Size.Height * (2 - y) / 4;

            g.DrawLine(myPen, OldX, OldY, NewX, NewY);

            OldX = NewX;
            OldY = NewY;
        }

        private void btnPlot_Click(object sender, EventArgs e)
        {
            float x = -10;
            int EqHandle, xHandle;

            g.Clear(PictureBox1.BackColor);

            xHandle = uc.ucDefineVariable("x");
            EqHandle = uc.ucParse(txtPlotExpression.Text);

            while (x <= 10)
            {
                uc.ucSetVariableValue(xHandle, x);
                CartesianLineTo(x, (float)uc.ucEvaluate(EqHandle));
                x = x + (float)0.125;
            }

            uc.ucReleaseItem(EqHandle);
            uc.ucReleaseItem(xHandle);
        }
               
        private void cmbNumericFormat_TextUpdate(object sender, EventArgs e)
        {
            NumericFormat = cmbNumericFormat.Text;
            if (NumericFormat == "Standard") NumericFormat = "N";
            if(NumericFormat == "Scientific") NumericFormat = "E";
            if(NumericFormat.ToUpper() != NumericFormat.ToLower()) NumericFormat = NumericFormat.Substring(0, 1);
        }

        private void cmbNumericFormat_SelectedIndexChanged(object sender, EventArgs e)
        {
            cmbNumericFormat_TextUpdate(sender, e);
        }
    }
}
