unit DemoDlp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ShellApi, ExtCtrls, StdCtrls, uCalcDlp; // Run Setup.Bat to copy over uCalc files.

type
  TForm1 = class(TForm)
    txtSumMax: TEdit;
    txtSumExpression: TEdit;
    txtPlotEq: TEdit;
    txtSumResult: TEdit;
    txtResult: TEdit;
    btnEval: TButton;
    btnDefine: TButton;
    btnSum: TButton;
    btnPlot: TButton;
    Label4: TLabel;
    Image1: TImage;
    lblElapsed: TLabel;
    lblFPU: TLabel;
    chkInvalidOp: TCheckBox;
    chkDenormalOp: TCheckBox;
    chkDivisionBy0: TCheckBox;
    chkOverflow: TCheckBox;
    chkUnderflow: TCheckBox;
    chkPrecisionLoss: TCheckBox;
    cmbExpression: TComboBox;
    cmbDefinition: TComboBox;
    cmbNumericFormat: TComboBox;
    procedure btnSumClick(Sender: TObject);
    procedure btnEvalClick(Sender: TObject);
    procedure btnDefineClick(Sender: TObject);
    procedure btnPlotClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure chkInvalidOpClick(Sender: TObject);
    procedure chkDenormalOpClick(Sender: TObject);
    procedure chkDivisionBy0Click(Sender: TObject);
    procedure chkOverflowClick(Sender: TObject);
    procedure chkUnderflowClick(Sender: TObject);
    procedure chkPrecisionLossClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  Form1: TForm1;
  H, W: Variant; { Image1 Height and Width }


implementation

{$R *.DFM}

procedure CartesianLineTo(a: Variant; b: Variant);
var
  x, y: Integer;
begin
  x := W * (a + 10) / 20;
  y := H * (2 - b) / 4;
  Form1.Image1.Canvas.LineTo(x, y);
end;


// uCalc callback routines

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure MyArea(Expr: Longword); stdcall;
var
   MyLength, MyWidth: Extended;
begin
   MyLength := ucArg(Expr, 1);
   MyWidth := ucArg(Expr, 2);
   
   if MyLength < 0 then ucRaiseErrorMessage(Expr, 'Length cannot be negative');
   if MyWidth  < 0 then ucRaiseErrorMessage(Expr, 'Width cannot be negative');

   ucReturn(Expr, MyLength * MyWidth);
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
function NonNativeFunc(a: Double; var b: Longint; c: Byte): Extended; stdcall;
begin
   NonNativeFunc := a + b + c;
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure MyIIF_Numeric(Expr: Longword); stdcall;
var
   Condition: Extended;
   TruePart, FalsePart: Longword;
begin
   Condition := ucArg(Expr, 1);
   TruePart  := ucArgHandle(Expr, 2);
   FalsePart := ucArgHandle(Expr, 3);

   if Condition <> 0 
   then ucReturn(Expr, ucEvaluate(TruePart)) 
   else ucReturn(Expr, ucEvaluate(FalsePart));
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure MyIIF_String(Expr: Longword); stdcall;
var
   Condition: Extended;
   TruePart, FalsePart: Longword;
begin
   Condition := ucArg(Expr, 1);
   TruePart  := ucArgHandle(Expr, 2);
   FalsePart := ucArgHandle(Expr, 3);

   if Condition <> 0 
   then ucReturnStr(Expr, ucEvaluateStr(TruePart))
   else ucReturnStr(Expr, ucEvaluateStr(FalsePart));
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure Native_MsgBox(Expr: Longword); stdcall;
begin
   ucReturnLng(Expr, MessageBox(0, PChar(string(ucArgStr(Expr, 1))), PChar(string(ucArgStr(Expr, 2))), ucArgLng(Expr, 3)));
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
function NonNative_MsgBox( Prompt: PAnsiChar; var Title: PAnsiChar; Buttons: Longint): Longint; stdcall;
begin
   NonNative_MsgBox := MessageBox(0, PChar(string(Prompt)), PChar(string(Title)), Buttons);
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure MyAverage(Expr: Longword); stdcall;
var
   x: Longint;
   Total: Extended;
begin
   Total := 0;
   for x := 1 to ucArgCount(Expr) do Total := Total + ucArg(Expr, x);
   ucReturn(Expr, Total / ucArgCount(Expr));
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure MyLeft(Expr: Longword); stdcall;
begin
   ucReturnStr(Expr, copy(ucArgStr(Expr, 1), 1, Trunc(ucArg(Expr, 2))));
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure StringRepeat(Expr: Longword); stdcall;
var
   x: Longint;
   TotalString: AnsiString;
begin
   TotalString := '';
   for x := 1 To Trunc(ucArg(Expr, 2))
   do TotalString := TotalString + ucArgStr(Expr, 1);
   ucReturnStr(Expr, TotalString);
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
var AlreadyDisplayedOnce: Longint = 0;
function MyErrorHandler(t: Longword): Longword; stdcall;
begin
   if AlreadyDisplayedOnce = ucFalse then
      ShowMessage('Error Handler message: ' + ucErrorMessage(0, t) + Chr(13)
         + 'Offending symbol: ' + ucErrorSymbol(t) + Chr(13)
         + 'Error Location: ' + IntToStr(ucErrorLocation(t)) + Chr(13) + Chr(13)
         + 'This message box will not be displayed for the next error.' + Chr(13)
         + 'Remove "AlreadyDisplayedOnce := ucTrue;" in the demo source code to change this.');
   
   // Remove the line below if you want the message box to be
   // displayed every time there's an error.
   AlreadyDisplayedOnce := ucTrue;
   MyErrorHandler := ucAbort;
end;

procedure MyNumericFormat(Expr: Longword); stdcall;
var
   fmt: string;
   Value: Extended;
begin
   Value := StrToFloat(ucArgStr(Expr, 1));
   fmt := '%g';
   if Form1.cmbNumericFormat.Text = 'Scientific' then fmt := '%e';
   if Form1.cmbNumericFormat.Text = 'Fixed'      then fmt := '%f';
   if Form1.cmbNumericFormat.Text = 'Number'     then fmt := '%n';
   if Form1.cmbNumericFormat.Text = 'Money'      then fmt := '%m';

   ucReturnStr(Expr, Format(fmt, [Value]));
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure ucSum(Expr: Longword); stdcall;
var
   Expression, VarHandle: Longword;
   Start, Finish, sStep: Extended;
   x, Total: Extended;
begin
   Expression := ucArgHandle(Expr, 1);
   Start      := ucArg(Expr, 2);
   Finish     := ucArg(Expr, 3);
   sStep      := ucArg(Expr, 4);
   VarHandle  := ucArgHandle(Expr, 5);

   x := Start;
   Total := 0;
   while x <= Finish do
   begin
      ucSetVariableValue(VarHandle, x);
      Total := Total + ucEvaluate(Expression);
      x := x + sStep;
   end;

   ucReturn(Expr, Total);
end;

// IMPORTANT: Always remember to use  stdcall  for uCalc callbacks
procedure ucSolve(Expr: Longword); stdcall;
var
   Expression, Variable, Iterations: Longword;
   a, b, fa, fb, Value, tmp: Double;
begin
   Expression := ucArgHandle(Expr, 1);
   a := ucArg(Expr, 2);
   b := ucArg(Expr, 3);
   Variable := ucArgHandle(Expr, 4);
   Iterations := 0;

   ucSetVariableValue(Variable, a); fa := ucEvaluate(Expression);
   ucSetVariableValue(Variable, b); fb := ucEvaluate(Expression);

   if fb < fa then
   begin  // swap a, b
      tmp := a;
      a := b;
      b := tmp;
   end;

   while Abs(b - a) > 0.000000000000001 do
   begin
      ucSetVariableValue(Variable, (a + b) / 2);

      Value := ucEvaluate(Expression);

      if Value = 0 then begin a := (a + b) / 2; Break; end;

      if Value < 0
      then a := (a + b) / 2
      else b := (a + b) / 2;

      Iterations := Iterations + 1;
      if Iterations = 100 then Break;
   end;

   If Abs(Value) > 0.0000000001 Then ucRaiseErrorMessage(Expr, 'Solution not found');

   ucReturn(Expr, a);
end;

procedure Test(Expr, Answer: string; t: Longword = 0);
begin
   if Answer <> ucEvalStr(Expr, t) then Application.MessageBox(PChar(Expr + Chr(10) + ucEvalStr(Expr, t) + Chr(10) + Answer), '', 0);
end;

procedure TestExpand(Expr, Answer: string; t: Longword = 0);
begin
   if Answer <> ucExpand(Expr, t) then Application.MessageBox(PChar(Expr + Chr(10) + ucExpand(Expr, t) + Chr(10) + Answer), '', 0);
end;

procedure TForm1.FormCreate(Sender: TObject);
begin
   Image1.Canvas.Rectangle(0, 0, Image1.Width, Image1.Height);

   // The definitions here are for "callback" routines.
   // Each line ends with the address of a routine, with the name of the routine preceded by @.
   // The actual code for these routines can be found elsewhere in this file.
   // Do a search for MyErrorHandler, for instance, to see the actual code for it.

   // The MyErrorHandler() function will be called whenever and wherever an
   // error (such as "Syntax Error") is raised.
   ucAddErrorHandler(@MyErrorHandler);

   // Lets user configure numeric output: Scientific, Fixed, Money, Standard, etc.
   ucSetOutput(@MyNumericFormat);

   // Native callback definitions, which start with "Native: " are recommended for
   // speed, and for routines that use uCalc's default numeric and string types.
   // MyArea() returns the product of multiplying the two arguments.
   // MyLeft() returns the left-most characters of a string.  The second argument represents the number of characters.
   // MsgBox() displays a message box.  The 1st argument is required.  The others are optional.
   ucDefineFunction('Native: MyArea(Length, Width)', @MyArea);
   ucDefineFunction('Native: MyLeft(Text As String, Count) As String', @MyLeft);
   ucDefineFunction('Native: MyMsgBox(Prompt As String, Title As String = "uCalc", Buttons = 0) As Long', @Native_MsgBox);
   ucDefineSyntax('MsgBox ::= MyMsgBox');

   // The three consecutive dots "..." mean that MyAverage can take any number of arguments
   // (however, because "x" is specified, there must be at least one argument).
   ucDefineFunction('Native: MyAverage(x ...)', @MyAverage);

   // Two versions of MyIIF can co-exist peacefully with the same name because
   // they are defined with different argument types (numeric vs string).
   // By passing the last two arguments ByExpr, the callback can chose to evaluate only one of of them.
   ucDefineFunction('Native: MyIIf(ByVal cond, ByExpr TruePart, ByExpr FalsePart)', @MyIIF_Numeric);
   ucDefineFunction('Native: MyIIf(ByVal cond, ByExpr TruePart As String, ByExpr FalsePart As String) As String', @MyIIF_String);

   // The following defines the "*" operator so that MyString * n returns MyString repeated n times.
   // For instance, "He " * 3, would return "He He He".  20 represents the precedence level.  It is
   // arbitrarily set at the same level as that of the "*" multiplication operator defined in uCalcDlp.Pas.
   ucDefineOperator('Native: Precedence("*") {MyString As String} * {Number} As String', @StringRepeat);

   // Non-native callbacks are intended for routines that were not created with
   // uCalc in mind, such as the Windows API, or other pre-existing DLL routines.
   // It is convenient (no need to adapt the routines; all you need is a function
   // address), but not as fast as Native callbacks.
   // NonNativeFunc() simply adds the three arguments.
   // nn_MsgBox is a Non-Native version of MsgBox, which displays a message box.
   ucDefineFunction('NonNativeFunc(ByVal a As Double, ByRef b As Long, ByVal c As Byte) As Extended', @NonNativeFunc);
   ucDefineFunction('NonNativeMsgBox(ByVal Prompt As String, ByRef Title As String = "uCalc", ByVal Buttons As Long = 0) As Long', @NonNative_MsgBox);

   // This definition is for a summation.  See the callback ucSum() routine for the
   // actual code, which runs a loop that adds up the total for the expression in
   // the first argument a number of times based on the second and third arguments.
   // For instance Sum(g^2+1, 1, 5, 1, g) returns 60 and Sum(x^2, 1, 10) returns 385.
   // The last two arguments are optional, so they default to 1 and x if omitted.
   //
   // The first argument is passed "ByExpr".  So instead of being evaluated before
   // being passed to the callback the way an ordinary argument would be, a handle
   // for the expression is passed so that the callback can evaluate it (in this
   // case numerous times).
   //
   // The last argument is passed "ByHandle".  This causes the callback to receive
   // a handle for the variable being passed, in such a way that it can be linked to
   // the summation counter, and integrated into the expression in the first argument.
   //
   // The actual function being defined is Sum_().  Then a syntax construct named
   // Sum() is defined in such a way that the last argument gets defined as a local
   // variable.  So if you evaluate Sum(x^2, 5, 10, 1, x), the local "x" in this
   // expression will not interfere with a pre-existing variable named x.  Also you
   // do not need to declare a variable ahead of time to use it as a counter for Sum().
   ucDefineFunction('Native: Sum_(ByExpr Expr, Start, Finish, Step, ByHandle Var)', @ucSum);
   ucDefineSyntax('Sum({Expr}, {Start}, {Finish} [, {Step=1} [, {Var=x}]])'
                + '::= Local({Var}, Sum_({Expr}, {Start}, {Finish}, {Step}, {Var}))');

   // The following routine solves an equation.
   // The callback code is based on the Bisection Method algorithm.
   // The concept here is very similar to that of Sum().
   // Two syntax constructs are defined.  The second one rearranges the equation
   // if it includes an equal sign.
   // For instance Solve(x^2 = 9+x) becomes Solve(x^2 - (9+x))
   // Solve(x^2 + 1 = 26) returns 5.
   // Solve(x^2 + 1 = 26, -1000, 0) returns -5.
   ucDefineFunction('Native: Solve_(ByExpr Expr, a, b, ByHandle Var)', @ucSolve);
   ucDefineSyntax('Solve({Expr} [, {a=-100000000} [, {b=100000000} [, {Var=x}]]]) '
                + '::= Local({Var}, Solve_({Expr}, {a}, {b}, {Var}))');
   ucDefineSyntax('Solve({Left} = {Right} [, {etc}]) ::= Solve({Left}-({Right}) {etc: , {etc}})');
end;

procedure TForm1.btnSumClick(Sender: TObject);
var
   ExprHandle, xHandle: Longword;
   x, SumMax, SumTotal: Extended;
   Code: Integer;
   TimerStart: TDateTime;
   Elapsed: TTimeStamp;
begin
   x := 1;
   SumTotal := 0;

   xHandle := ucDefineVariable('x', @x);
   ExprHandle := ucParse(txtSumExpression.Text);

   val(txtSumMax.Text, SumMax, Code);
   TimerStart := Time;

   while x <= SumMax do
   begin
     SumTotal := SumTotal + ucEvaluate(ExprHandle);
     x := x + 1;
   end;

   Elapsed := DateTimeToTimeStamp(Time - TimerStart);
   
   lblElapsed.Caption := 'Elapsed Time: ' + FloatToStr(Elapsed.Time/1000) + ' seconds';
   txtSumResult.Text  := FormatFloat('#', SumTotal);
   
   ucReleaseItem(ExprHandle);
   ucReleaseItem(xHandle);
end;

procedure TForm1.btnEvalClick(Sender: TObject);
begin
   // ucEval() evaluates an expression and returns a numeric result.
   // ucEvalStr() can evaluate expressions of any data type (numeric, string, or other).
   // If the result is numeric ucEvalStr conveniently converts it to a string for you.
   txtResult.Text := ucEvalStr(cmbExpression.Text);
end;

procedure TForm1.btnDefineClick(Sender: TObject);
begin
   // ucDefine can define a variable, function, operator, syntax construct,
   // constant, pattern, data type, and more.
   //
   // Examples:
   //
   // ucDefine('Var: MyVariable As String');
   // ucDefine('Func: Area(Length, Width) = Length * Width');
   // ucDefine('Op: 50 {x}%  :=  x / 100');   // 50 is an arbitrary precedence level.
   // ucDefine('Syntax: {a} + {a} ::= "Two {a}s are better than one {a}"');
   // ucDefine('Const: Pi = Atan(1) * 4');
   ucDefine(cmbDefinition.Text);
end;

procedure TForm1.btnPlotClick(Sender: TObject);
var
   x: Extended;
   EqHandle, xHandle: LongWord;
begin
   Image1.Canvas.FillRect(ClientRect);
   xHandle := ucDefineVariable('x', @x);
   EqHandle := ucParse(txtPlotEq.Text);

   W := Image1.Width;
   H := Image1.Height;
   x := -10;

   Image1.Canvas.MoveTo(0, H/2);

   while x <= 10 do
   begin try
      CartesianLineTo(x, ucEvaluate(EqHandle));
   except end;
      x := x + 0.125;
   end;

   ucReleaseItem(EqHandle);
   ucReleaseItem(xHandle)
end;

procedure TForm1.chkInvalidOpClick(Sender: TObject);
begin
   // If toggled, then an expression such as 0 / 0 raises an error.
   // If toggled again, then 0 / 0 returns NaN.
   ucFPU(uc_ToggleFPU, uc_FPU_Mask_InvalidOp);
end;

procedure TForm1.chkDenormalOpClick(Sender: TObject);
begin
   ucFPU(uc_ToggleFPU, uc_FPU_Mask_DenormalOp);
end;

procedure TForm1.chkDivisionBy0Click(Sender: TObject);
begin
   // If toggled, then 1 / 0 raises an error.
   // If toggled again, then 1 / 0 returns Inf.
   ucFPU(uc_ToggleFPU, uc_FPU_Mask_ZeroDivide);
end;

procedure TForm1.chkOverflowClick(Sender: TObject);
begin
   // If toggled, then 100! ^ 150 raises an error.
   // If toggled again, then 100! ^ 150 returns Inf.
   ucFPU(uc_ToggleFPU, uc_FPU_Mask_Overflow);
end;

procedure TForm1.chkUnderflowClick(Sender: TObject);
begin
   // If toggled, then (1/100!)^150 raises an error.
   // If toggled again, then (1/100!)^150 returns 0.
   ucFPU(uc_ToggleFPU, uc_FPU_Mask_Underflow);
end;

procedure TForm1.chkPrecisionLossClick(Sender: TObject);
begin
   // If toggled, an expression that cannot be represented in exact form
   // such as "1/3" will raise an error.
   // If toggled again, "1/3" returns the closest extended precision approximation.
   ucFPU(uc_ToggleFPU, uc_FPU_Mask_PrecisionLoss);
end;

end.
